; *****************************************************************************
;        Software License Agreement				    
;								    
; The software supplied herewith by Microchip Technology 	    
; Incorporated (the Company) for its PICmicro Microcontroller is 
; intended and supplied to you, the Companys customer, for use     
; solely and exclusively on Microchip PICmicro Microcontroller	    
; products. The software is owned by the Company and/or its         
; supplier, and is protected under applicable copyright laws. All   
; rights are reserved. Any use in violation of the foregoing 	    
; restrictions may subject the user to criminal sanctions under	    
; applicable laws, as well as to civil liability for the breach of  
; the terms and conditions of this license.			    
;								    
; THIS SOFTWARE IS PROVIDED IN AN AS IS CONDITION. NO WARRANTIES, 
; WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED 
; TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 	    
; PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT, 
; IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR 	    
; CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.		    
;								    
;								    
; Bootloader for the PIC18F252/452
;
; DESCRIPTION:
; This is an example of a bootloader routine for the PIC18FX52. The 
; PICDEM2 was used for development. A 10MHz clock was selected.
;
; INSTRUCTIONS:
; Press button RA4 and reset on the PICDEM2. Release reset. Then 
; release button RA4. Now the device is waiting for a text data stream
; to be parced and loaded into the part. 
;
; Use 'Hyperterm' to download the HEX file to be bootloaded. Set 
; the terminal for 57.6kb, 8, N, 1, and enable XON/XOFF. Use the 
; 'Send Text File' option to transfer the HEX file. The HEX file
; must be INTEL32 HEX format.
;
; RB0 will light if an error occured.
; RB1 will light if the code loaded successfully and an EOL was received.
; RB2 will toggle for each HEX line recieved.
;
;
; GENERAL INFO:
; When writing code to be bootloaded, be aware of the re-mapping of
; the reset and interrupt vectors.
;
; Memory Map
;	-----------------
;	|    0x0000	|		
;   	|		|
;	|  Boot Block 	|	(this program)
; 	|		|
;	|    0x0200	|	Re-mapped Reset Vector
;	|    0x0208	|	Re-mapped High Priority Interrupt Vector
;	|    0x0218	|	Re-mapped Low Priority Interrupt Vector
;	|		|
;	|	|	|
;	|		|
;    	|  Code Space 	|
;	|		|
;	|	|	|
;	|		|
;	|    0x7FFE     |
;	-----------------
; *****************************************************************************

 

; *****************************************************************************
	#include P18F452.INC		; Standard include
; *****************************************************************************



; *****************************************************************************
#define BRG_VAL		D'10'		; Desired baud rate
; *****************************************************************************



; *****************************************************************************
_BOOTLOAD	UDATA_ACS

DATACNT		res	1		; Storage area for INHX32
ADDRESS_H	res	1
ADDRESS_L	res	1
RECORD		res	1
DATA_BUFF	res	0x10

LAST_BYTE	res	1

CHKSUM		res	1		; Checksum accumulator

COUNTER		res	1		; General counter

TEMP1		res	1		; Temp storage

RXDATA		res	1		; Receive storage area

ADDRESS_UH	res	1		; Upper word for 32-bit HEX
ADDRESS_UL	res	1
; *****************************************************************************



; *****************************************************************************
_V_RESET	CODE	0x0000
	btfss	PORTA, 4		; Enter bootloader if button 
	bra	Setup			;  RA4 is pressed.
	bra	0x0200
; *****************************************************************************



; *****************************************************************************
_V_INT_HIGH	CODE	0x0008
	bra	0x0208			; Re-map Interrupt vector
; *****************************************************************************



; *****************************************************************************
_V_INT_LOW	CODE	0x0018
	bra	0x0218			; Re-map Interrupt vector
; *****************************************************************************


; *****************************************************************************
_V_ERROR	CODE	0x001A		; Bootload error vector

GeneralErr
	bsf	PORTB, 0		; Indicate some error
	bra	$

EndOfRecord
	bsf	PORTB, 1		; Indicate successful write		
	bra	$
; *****************************************************************************



; *****************************************************************************
_V_SETUP	CODE
Setup
	btfss	PORTA, 4		; Wait for button release
	bra	Setup

	clrf	LATB
	clrf	TRISB
	
	bcf	TRISC, 6		; Setup tx pin
	bsf	TRISC, 7		; Setup rx pin

	movlw	b'10010000'		; Setup rx and tx
	movwf	RCSTA
	movlw	b'00100110'		
	movwf	TXSTA

	movlw	BRG_VAL			; Setup the baud rate	
	movwf	SPBRG

	movf	RCREG, W		; Empty the buffer
	movf	RCREG, W
; *****************************************************************************



; *****************************************************************************
; This section reads and parses one line of INTEL32 HEX. 
; 

StartOfLine
	movlw	D'17'			; XON, enable data stream
	rcall	WrRS232


	btg	LATB, 2

WaitForStart
	rcall	GetChar			; Look for a start of line
	xorlw	':'
	bnz	WaitForStart		; Go back for another character


	rcall	GetByte			; Get the data count
	movwf	CHKSUM			
	movwf	DATACNT			; Set the data counter
	addlw	0x03
	movwf	COUNTER		
	

	lfsr	0, ADDRESS_H	
GetNextDat				; Get the data
	rcall	GetByte
	addwf	CHKSUM, F
	movwf	POSTINC0
	decfsz	COUNTER, F
	bra	GetNextDat

	rcall	GetByte			; Get the checksum
	movwf	LAST_BYTE


	movlw	D'19'			; XOFF, disable data stream
	rcall	WrRS232
; *****************************************************************************



; *****************************************************************************
; This tests, writes, and verifies the line of HEX data.
;
; Notes:
; The HEX data is assumed to be padded to fill a 16 byte 
; INHX32 line.
;
; Programming is limited to program memory space only.
;
; Not all record types are decoded.

TestChkSum				; Examine the checksum
	movf	LAST_BYTE, W
	negf	CHKSUM
	xorwf	CHKSUM, W
	bnz	GeneralErr		; Stop if an error is detected


TestRecordType				; Check the record type
	movlw	0x00			
	xorwf	RECORD, W
	bz	NormalDataRecord	; If a normal record then process
	movlw	0x01
	xorwf	RECORD, W
	bz	EndOfRecord		; IF EOF then finish
	movlw	0x04
	xorwf	RECORD, W
	bz	ExtendedRecord		; If extended then adjust
	bra	StartOfLine		; Otherwise, undefined record type...


ExtendedRecord				; Read in upper word of 32-bit address
	movff	DATA_BUFF, ADDRESS_UH	
	movff	DATA_BUFF + 1, ADDRESS_UL	

	bra	StartOfLine


NormalDataRecord			; Normal data record

	movf	ADDRESS_UH		; Get the next line if too high
	bnz	StartOfLine

	movlw	0x20			; Write ID locations?
	xorwf	ADDRESS_UL, W
	bz	WriteIDLocations
	movlw	0x30			; Write Config locations?
	xorwf	ADDRESS_UL, W
	bz	WriteConfig
	movlw	0xF0			; Write Data EEPROM?
	xorwf	ADDRESS_UL, W
	bz	WriteEEPROM
	
	btfsc	ADDRESS_H, 7		; Get the next line if too high
	bra	StartOfLine

	movlw	0x02			; Prevent writing to boot area
	subwf	ADDRESS_H, W
	bnc	StartOfLine


EraseRow				; Erase row if pointing to first byte in row
	rcall	SetPointers
	movf	ADDRESS_L, W		; Check for row allignment
	andlw	b'00111111'
	bnz	TestForBlankMem
	movlw	b'10010100'		; Setup erase
	movwf	EECON1
	rcall	StartWrite		; Erase the row				
	 

TestForBlankMem				; Do not write unless blank
	rcall	SetPointers
TestNext
	tblrd	*+			; Read 
	comf	TABLAT			; Test for blank (0xFF)
	bnz	GeneralErr		
	decfsz	COUNTER
	bra	TestNext


Write2Blocks				; Program one HEX line (2 blocks)
	rcall	SetPointers
	rcall	WriteBlock		; Write 2 blocks (16 bytes)
	rcall	WriteBlock


VerifyWrite				; Verify the data		
	rcall	SetPointers
VerifyNext				; Compare
	tblrd	*+
	movf	POSTINC2, W
	xorwf	TABLAT
	bnz	GeneralErr		; Stop if bad verify
	decfsz	COUNTER
	bra	VerifyNext

	bra	StartOfLine		; Go get the next line of data
; *****************************************************************************



; *****************************************************************************
WriteIDLocations
	movff	ADDRESS_UL, TBLPTRU
	rcall	SetPointers
	rcall	WriteBlock
	bra	StartOfLine
; *****************************************************************************



; *****************************************************************************
WriteConfig
	movlw	b'11000100'		; Setup to write fuses
	movwf	EECON1

	rcall	SetPointers
	movff	ADDRESS_UL, TBLPTRU
	movff	DATACNT, COUNTER

WriteNextConfig
	
	movf	POSTINC2, W
	movwf	TABLAT
	tblwt	*
	rcall	StartWrite		; Initiate a write
	
	tblrd 	*+
	decfsz	COUNTER, F
	bra	WriteNextConfig

	bra	StartOfLine
; *****************************************************************************



; *****************************************************************************
WriteEEPROM
	movlw	b'00000100'		; Setup to write DataEE
	movwf	EECON1

	rrcf	ADDRESS_H, W
	rrcf	ADDRESS_L, W
	movwf	EEADR

	rcall	SetPointers
	movlw	0x08
	movwf	COUNTER
	
WriteNextEE
	movff	POSTINC2, EEDATA	; Load the data
	movf	POSTINC2, W

	rcall	StartWrite		; Initiate a write

	btfsc	EECON1, WR		; Wait for the write to finish
	bra	$ - 2

	incf	EEADR, F
	decfsz	COUNTER, F
	bra	WriteNextEE

	bra	StartOfLine
; *****************************************************************************
					






; *****************************************************************************
; Reset pointers to program mem and data mem.

_F_SET_PNTR	CODE

SetPointers
	movff	ADDRESS_H, TBLPTRH	; Reset pointer
	movff	ADDRESS_L, TBLPTRL
		
	lfsr	2, DATA_BUFF

	movlw	0x10
	movwf	COUNTER

	return
; *****************************************************************************



; *****************************************************************************
; Write a byte to the serial port.

_F_WRT_RS232	CODE

WrRS232
	btfss	PIR1, TXIF		; Write only if TXREG is ready
	bra	WrRS232
	
	movwf	TXREG
	return
; *****************************************************************************



; *****************************************************************************
; Get a character from the serial port.

_F_GET_CHAR	CODE

GetChar
	btfss	PIR1, RCIF
	bra	GetChar			; Retry if no data

	movf	RCREG, W		; Condition the data
	bcf	WREG, 7
	movwf	RXDATA

;	rcall	WrRS232			; Debug, echo data back

	return
; *****************************************************************************



; *****************************************************************************
; Get a byte of data from the serial port.

_F_GET_BYTE	CODE

GetByte
	rcall	GetChar			; Get a character
	rcall	Ascii2Hex		; Convert to hexdecimal
	swapf	RXDATA, W
	movwf	TEMP1

	rcall	GetChar			; Get the next character
	rcall	Ascii2Hex		; Convert to hexdecimal
	movf	TEMP1, W

	iorwf	RXDATA, W		; Assemble the two nibbles
	
	return
; *****************************************************************************



; *****************************************************************************
; Convert ASCII to HEX.

_F_ASCII_HEX	CODE

Ascii2Hex				; Convert ASCII to Hexdecimal
	movlw	'0'
	subwf	RXDATA
	movlw	0xF0
	andwf	RXDATA, W
	bz	A2HDone			; If no upper nibble then number only
	movlw	'A'-'0'-0x0A		; else get the letter
	subwf	RXDATA, F
A2HDone	return
; *****************************************************************************



; *****************************************************************************
; Write a block of data to program memory.

_F_WRT_BLOCK	CODE

WriteBlock
	movlw	0x08

Lp1	movff	POSTINC2, TABLAT	; Load the holding registers
	tblwt	*+
	decfsz	WREG, F
	bra	Lp1

	tblrd	*-			; Point back into the block

	movlw	b'10000100'		; Setup writes
	movwf	EECON1

	rcall	StartWrite		; Write the data

	tblrd	*+			; Point to the beginning of the next block

	return
; *****************************************************************************	



; *****************************************************************************
; Unlock and start the write or erase sequence.

_F_START_WRITE

StartWrite
	movlw	0x55			; Unlock
	movwf	EECON2
	movlw	0xAA
	movwf	EECON2
	bsf	EECON1, WR		; Start the write
	nop

	return
; *****************************************************************************





; *****************************************************************************
; Automatic block aligning function with padding
;WriteBlock
;	call	FlushBuffer
;
;	movlw	0x07			; Determine how much of the block to write
;	andwf	TBLPTRL, W
;
;	comf	WREG, F		
;	andlw	0x07
;	incf	WREG, F
;
;Lp1	movff	POSTINC2, TABLAT	; Load the holding registers
;	tblwt	*+
;	decfsz	WREG, F
;	bra	Lp1
;
;	tblrd	*-			; Point back into the block
;
;	rcall	StartWrite		; Write the data
;
;	tblrd	*+			; Point to the beginning of the next block
;
;	return
;
; *****************************************************************************	



; *****************************************************************************
;FlushBuffer				; Clear any data in the holding registers
;
;	movff	TBLPTRL, PRODL		; Copy the LSB of the pointer
;
;	clrf	TBLPTRL
;	setf	TABLAT
;	movlw	0x08			; Clear the buffer
;Lp2	tblwt	*+
;	decfsz	WREG, F
;	bra	Lp2
;
;	movff	PRODL, TBLPTRL		; Restore the LSB of the pointer
;
;	return
; *****************************************************************************





	END
