; PICkit 3 Starter Kit
; Assembly Lesson 11 - Indirect addressing
;
; This lesson covers a very important topic of indirect addressing. The code uses indirect
; addressing to implement a moving average filter. This lesson adds a moving average 
; filter to the Analog-to-Digital code in Lesson 4. The moving average keeps a list of the 
; last ADC values (n) and averages them together. The filter needs two parts: A circular 
; queue and a function to calculate the average. 
;
; Twisting the potentiometer changes the value read by the Analog-to-Digital converter. 
; The filtered value is then sent to the LED display.
;
; This lesson provides the same outcome as Lesson 4. The user rotates the POT to see
; the LEDs rotate. The top four MSbs of the ADC value are reflected onto the LEDs.
;

; VSK @THU 18.04.2023
; MPLABX v6.00 / pic-as v2.40
; Board: PICkit 3 Low Pin Count Demo Board
; *******************************************************************
; * See Low Pin Count Demo Board User's Guide for Lesson Information*
; *******************************************************************

// config statements in Config_16F1829.s
#include <xc.inc>
  
GLOBAL Display,Queue,RunningSum,Round,temp ;make them global -> watchable when debugging
PSECT udata_shr     ;shared memory location that is accessible from all banks
Display:
    DS      1       ;reserve 1 byte
Queue:
    DS      8       ;reserve 8 byte
RunningSum:
    DS      2       ;reserve 2 byte
Round:
    DS      2       ;reserve 2 byte
temp:
    DS      1       ;reserve 1 byte


PSECT resetVec,class=CODE,delta=2   ;define "-presetVec=0h" in custom linker options
resetVec:                       ;Setup main init
    banksel     OSCCON          ;bank1
    movlw       00111000B       ;set cpu clock speed of 500KHz
    movwf       OSCCON          ;move contents of the working register into OSCCON

                                ;Configure the LEDs
    clrf        TRISC           ;make all of PORTC an output
    banksel     LATC            ;bank2
    clrf        LATC            ;start with all LEDs off

                                ;Configure the ADC/Poti
    banksel     TRISA           ;bank1
    bsf         TRISA, 4        ;Poti is connected to RA4....set as input
    movlw       00001101B       ;select RA4 as source of ADC and enable the module (carefull, this is actually AN3)
    movwf       ADCON0
    movlw       00010000B       ;left justified - Fosc/8 speed - vref is Vdd
    movwf       ADCON1
    banksel     ANSELA          ;bank3
    bsf         ANSELA, 4       ;analog for ADC

                                ;Init the Filter
    call        FilterInit      ;init the moving average filter
MainLoop:
    call        A2d             ;start the ADC (value -> w)
    call        Filter          ;send it to the filter
    movwf       Display         ;save the filtered value
    swapf       Display,w       ;swap the nybbles to put the high order
    banksel     LATC            ;bank2
    movwf       LATC            ;into the low order nybble on Port C
    goto        MainLoop

A2d:
                                ;Start the ADC
    nop                         ;requried ADC delay of 8uS => (1/(Fosc/4)) = (1/(500KHz/4)) = 8uS
    banksel     ADCON0          ;bank1
    bsf         ADGO            ;start the ADC
    btfsc       ADGO            ;this bit will be cleared when the conversion is complete
    goto        $-1             ;keep checking the above line until GO bit is clear
    movf        ADRESH, w       ;Get the top 8 MSbs (remember that the ADC result is LEFT justified!)

    return

                                ;keeps a running sum.
                                ;Before inserting a new value into the queue, the oldest is subtracted
                                ;from the running sum.  Then the new value is inserted into the array
                                ;and added to the running sum.
                                ;Assumes the FSR is not corrupted elsewhere in the program.  If the FSR
                                ;may be used elsewhere, this module should maintain a copy for it's
                                ;own use and reload the FSR before use.
FilterInit:
                                ;the 'high' and 'low' operators are used to return one byte of a multi-byte
                                ;label value. This is done to handle dynamic pointer calculations. Recall that
                                ;FSR0 is 16 bits wide (2 bytes).
    movlw          low Queue    ;point to the Queue holding the ADC values
    movwf          FSR0L
    movlw          high Queue   ;point to the Queue holding the ADC values
    movwf          FSR0H
    clrf           RunningSum
    clrf           RunningSum+1
    clrf           Queue
    clrf           Queue+1
    clrf           Queue+2
    clrf           Queue+3
    clrf           Queue+4
    clrf           Queue+5
    clrf           Queue+6
    clrf           Queue+7
    return
                                ;We already know the address of 'Queue' from the above 'cblock'. FSRO will
                                ;be pointing to the address of 0x001, hence the 'high' byte should be 0x000
                                ;and only the lower byte needs to be incremented. Still good practice to check
                                ;for overflows, however.
Filter:
    movwf       temp            ;save w (ADC value)

    movf        INDF0,w         ;subtract the current out of the sum. INDFO holds the value that FSRO pair is pointing to!
    subwf       RunningSum,f
    btfss       CARRY           ;was there a borrow?
    decf        RunningSum+1,f  ;yes, take it from the high order byte

    movf        temp,w
    movwf       INDF0           ;store in table
    addwf       RunningSum,f    ;Add into the sum
    btfsc       CARRY
    incf        RunningSum+1,f

    incf        FSR0L, f        ;incrememnt to next byte offset in queue
    btfsc       CARRY	  ;did it cause a carry? (should not happen in this demo snippet)
    incf        FSR0H, f	  ;yes, add it to the high byte of FSR0

    movf        FSR0L, w         ;VSK "L" get current offset into queue (this will load WREG with the address that FSR0 is pointing to!)
    xorlw       Queue+8         ;did it overflow out of the size of queue? (XOR the address of the last byte in the Queue with what FSR0 is pointing to)
    btfsc       ZERO            ;if FSR0 is pointing to the last address in the Queue (byte 8), then reset it
    call        FilterAssign    ;yes: reset the pointer to the 0 byte in the Queue

    bcf         CARRY           ;clear the carry
    rrf         RunningSum+1, w
    movwf       Round+1
    rrf         RunningSum, w   ;divide by 2 and copy to a version we can corrupt
    movwf       Round

    bcf         CARRY           ;clear the carry
    rrf         Round+1, f
    rrf         Round, f        ;divide by 4

    bcf         CARRY           ;clear the carry
    rrf         Round+1, f
    rrf         Round, f        ;divide by 8

    btfsc       CARRY           ;use the carry bit to round
    incf        Round, f
    movf        Round, w        ;load Wreg with the answer
    return

FilterAssign:
    movlw       low Queue       ;point to the Queue holding the ADC values
    movwf       FSR0L
    movlw       high Queue      ;point to the Queue holding the ADC values
    movwf       FSR0H
    return

    END resetVec
    