; PICkit 3 Starter Kit
; Assembly Lesson 12 - Table lookup
;
; This shows using a table lookup function to implement a
; binary to gray code conversion.  The POT is read by the A2D,
; The high order 4 bits then are converted to Gray Code and
; displayed on the LEDs.

; The Table Pointer is used to read a single byte in program memory
; The PC is also used to return a value in program memory
;
; Gray coded binary will be reflected on the LEDs in accordance with the POT reading
;

; VSK @THU 18.04.2023
; MPLABX v6.00 / pic-as v2.40
; Board: PICkit 3 Low Pin Count Demo Board
; *******************************************************************
; * See Low Pin Count Demo Board User's Guide for Lesson Information*
; *******************************************************************

// config statements in Config_16F1829.s
#include <xc.inc>
    
    
;Only uncomment one of these!!
;#define     __EnhancedMethod
;#define     __ClassicMethod
;#define_IS NOT WORKING CORRECTLY WHY?     __FSRMethod           
#define      __FSRMethod           
;#define     __TableRead

GLOBAL temp         ;make them global -> watchable when debugging
PSECT udata_shr     ;shared memory location that is accessible from all banks
temp:
    DS      1       ;reserve 1 byte


PSECT resetVec,class=CODE,delta=2   ; define "-presetVec=0h" in custom linker options
resetVec:                       ;Setup main init
    banksel    OSCCON           ;bank1
    movlw      00111000B        ;set cpu clock speed of 500KHz
    movwf      OSCCON           ;move contents of the working register into OSCCON

                                ;Configure the LEDs
    clrf       TRISC            ;make all of PORTC an output
    banksel    LATC             ;bank2
    clrf       LATC             ;start with all LEDs off

                                ;Configure the ADC/Potentimator
    banksel    TRISA            ;bank1
    bsf        TRISA, 4         ;Potentimator is connected to RA4....set as input
    movlw      00001101B        ;select RA4 as source of ADC and enable the module (carefull, this is actually AN3)
    movwf      ADCON0
    movlw      00010000B        ;left justified - Fosc/8 speed - vref is Vdd
    movwf      ADCON1
    banksel    ANSELA           ;bank3
    bsf        ANSELA, 4        ;analog for ADC
                                
    clrf       temp             ;Clear the RAM

MainLoop:
    call        A2d             ;get the ADC result
    swapf       ADRESH, w       ;move the high nibble to wreg for LED display
    call        BinaryToGrayCode;convert to Grey Code (Note: important that this is CALLED and not BRANCHED to since
                                ;the 'RETLW' instruction will return here with the conversion in wreg
    banksel     LATC            ;bank2
    movwf       LATC            ;move grey code into LEDs
    banksel     0               ;bank0
    bra         MainLoop        ;do this forever

A2d:                            ;Start the ADC
    nop                         ;requried ADC delay of 8uS => (1/(Fosc/4)) = (1/(500KHz/4)) = 8uS
    banksel     ADCON0          ;bank1
    bsf         ADGO            ;start the ADC
    btfsc       ADGO            ;this bit will be cleared when the conversion is complete
    goto        $-1             ;keep checking the above line until GO bit is clear
    return

BinaryToGrayCode:
    andlw       0x0F            ;mask off invalid entries
    movwf       temp
                                ;an #ifdef is a conditional symbol with is evaluated during preprocessing
                                ;the top of this program will include 4 #define statements which if uncommented, this
                                ;statement would become true. and the lines of code inbetween the #ifdef and #endif will
                                ;be evaluated
;VSK? #ifdef  __EnhancedMethod
;    bra         EnhancedMethod
;#endif
#ifdef __FSRMethod
    movlw       high TableStart ;get high order part of the beginning of the table
    movwf       FSR0H
    bsf         FSR0H,7         ; added VSK (for addressing flash memory)
    movlw       low TableStart  ;get lower order part of the table
    addwf       temp, w         ;add offset from ADC conversion
    btfsc       CARRY           ;did it overflow?
    incf        FSR0H, f        ;yes: increment high byte
    movwf       FSR0L           ; modify lower byte
    moviw       0[FSR0]         ;move the value that FSR0 is pointing to into wreg
    return                      ;grey code now in wreg, return to MainLoop
#endif
    
#ifdef __ClassicMethod
                                ;Using the Program Counter (PC) directly
    movlw       high TableStart ;get high order part of the beginning of the table
    movwf       PCLATH
    movlw       low TableStart  ;load starting address of table
    addwf       temp,w          ;add offset from ADC conversion
    btfsc       CARRY           ;did it overflow?
    incf        PCLATH,f        ;yes: increment PCLATH
    movwf       PCL             ;modify PCL
#endif
    
#ifdef __TableRead
    banksel     EEADRL          ;Select Bank for EEPROM registers

    movlw       high TableStart
    movwf       EEADRH          ;Store MSb of address
    movlw       low TableStart
    addwf       temp, w
    btfsc       CARRY           ;did it overflow?
    incf        EEADRH, f
    movwf       EEADRL          ;Store LSB of address

    bcf         CFGS            ;Do not select Configuration Space
    bsf         EEPGD           ;Select Program Memory
    bcf         GIE             ;Disable interrupts
    bsf         RD              ;Initiate read
    nop                         ;Executed
    nop                         ;Ignored
    bsf         GIE             ;Restore interrupts
    movf        EEDATL, w       ;Get LSB of word
    return                      ;return to main loop with the result in wreg
    ;movwf       PROG_DATA_LO    ; Store in user location
    ;movf        EEDATH,W        ; Get MSb of word
    ;movwf       PROG_DATA_HI    ; Store in user location
#endif
 
#ifdef __EnhancedMethod
EnhancedMethod:             ;using the newer (enhanced midrange only) method instead of dealing with PCL directly
    brw                     ;branches to the amount currently in wreg (e.g. if wreg = 0x04, we would return with a code
                            ;of b'0110', or decimal 4
#endif
                            ;Use the PIC's flash memory for values that do not change during runtime.
                            ;This helps to conserve the RAM and is good programming practice
TableStart:
    retlw       0000B       ; 0
    retlw       0001B       ; 1
    retlw       0011B       ; 2
    retlw       0010B       ; 3
    retlw       0110B       ; 4
    retlw       0111B       ; 5
    retlw       0101B       ; 6
    retlw       0100B       ; 7
    retlw       1100B       ; 8
    retlw       1101B       ; 9
    retlw       1111B       ; 10
    retlw       1110B       ; 11
    retlw       1010B       ; 12
    retlw       1011B       ; 13
    retlw       1001B       ; 14
    retlw       1000B       ; 15

    END resetVec
