/**
 *******************************************************************
 * Lesson 13 - "EEPROM"
 *
 * This lesson will provide code for writing and reading a single byte onto
 * the on-board EEPROM. EEPROM is non-volatile memory, meaning that it does
 * not lose its value when power is shut off. This is unlike RAM, which will
 * lose its value when no power is applied. The EEPROM is useful for storing
 * variables that must still be present during no power.
 * It is also convenient to use if the entire RAM space is used up.
 * Writes and reads to the EEPROM are practically instant and are much faster
 * than program memory operations.

 * Press the switch to save the LED state and then disconnect the power. When
 * power is then applied again, the program will start with that same LED lit.

 * When the lesson is first programmed, no LEDs will light up even with movement
 * of the POT. When the switch is pressed, the corresponding LED will be lit and
 * then the PIC will go to sleep until the switch is pressed again. Each press of
 * the switch saves the ADC value into EEPROM. The PIC uses interrupts to wake up
 * from sleep, take an ADC reading, save to EEPROM, and then goes back to sleep.
 *
 * PIC: 18F14K22
 * Compiler: XC8 v1.00  VSK 2.46
 * IDE: MPLABX v1.10    VSK 6.20
 *
 * Board: PICkit 3 Low Pin Count Demo Board
 * Date: 6.1.2012       VSK 19.09.25
 *
 * *******************************************************************
 * See Low Pin Count Demo Board User's Guide for Lesson Information*
 * ******************************************************************
 */

//VSK #include <htc.h>                         //PIC hardware mapping
#include <xc.h>                         //PIC hardware mapping
#define _XTAL_FREQ 500000               //Used by the compiler for the delay_ms(x) macro

#define DOWN        0
#define UP          1

#define SWITCH      PORTAbits.RA2

//VSK moved config bits to PICkit_3_Starter_Config.c

unsigned char adc(void);                    //prototype

    /* -------------------LATC-----------------
     * Bit#:  -7---6---5---4---3---2---1---0---
     * LED:   ---------------|DS4|DS3|DS2|DS1|-
     *-----------------------------------------
     */

void main(void) {
    OSCCON = 0b00100010;                 //500KHz clock speed
    TRISC = 0;                           //all LED pins are outputs
    LATC = 0;                            //init all LEDs OFF

    TRISAbits.TRISA2 = 1;                //switch input
    ANSELbits.ANS2 = 0;                  //digital input for switch

    TRISAbits.TRISA4 = 1;                //Potentiamtor is connected to RA4...set as input
    ANSELbits.ANS3 = 1;                  //analog input - different than pic16 syntax
    ADCON0 = 0b00001101;                 //select RA4 as source of ADC, which is AN3, and enable the module
    ADCON2 = 0b00000001;                 //left justified - FOSC/8 speed

    //setup interrupt on change for the switch
    INTCONbits.RABIE = 1;               //enable interrupt on change global
    IOCAbits.IOCA2 = 1;                 //when SW1 is pressed/released, enter the ISR

    RCONbits.IPEN = 0;                  //disable interrupt priorites
    INTCONbits.GIE = 1;                 //enable global interupts

    while (1) {
        LATC = eeprom_read(0x00);       //load whatever is in EEPROM to the LATCH
        SLEEP();                        //sleep until button is pressed
    }
}

unsigned char adc(void) {
//VSK    __delay_us(5);                  //wait for ADC charging cap to settle
//VSK always same channel here and function call waste lot of time anyway
    ADCON0bits.GO = 1;
    while (ADCON0bits.NOT_DONE) continue;//wait for conversion to be finished

    return ADRESH;                      //grab the top 8 MSbs
}

//VSK   void interrupt ISR(void) {
void __interrupt(high_priority) high_isr(void)
{
    unsigned char adc_value = 0;
    
    if (INTCONbits.RABIF) {             //SW1 was just pressed
        INTCONbits.RABIF = 0;           //must clear the flag in software
        __delay_ms(5);                  //debounce by waiting and seeing if still held down
        if (SWITCH == DOWN) {
            adc_value = adc();          //get the ADC value from the POT
            adc_value >>= 4;            //save only the top 4 MSbs

            LATC = adc_value;
                                        //EEPROM is non-volatile, meaning that it can retain its value when no power is applied
            eeprom_write(0x00, adc_value); //save the value to EEPROM and go back to sleep!
        }
    }
}


//VSK added EEPROM routines from old PLib
//##############################################################################
void Busy_eep ( void )
{
	while(EECON1bits.WR);
}
unsigned char Read_b_eep( unsigned int badd )
{
	EEADR = (badd & 0x0ff);
  	EECON1bits.CFGS = 0;
	EECON1bits.EEPGD = 0;
	EECON1bits.RD = 1;
	Nop();							//Nop may be required for latency at high frequencies
	Nop();							//Nop may be required for latency at high frequencies
	return ( EEDATA );              // return with read byte
}
void Write_b_eep( unsigned int badd,unsigned char bdata )
{
	char GIE_BIT_VAL = 0;
	EEADR = (badd & 0x0ff);
  	EEDATA = bdata;
  	EECON1bits.EEPGD = 0;
	EECON1bits.CFGS = 0;
	EECON1bits.WREN = 1;
	GIE_BIT_VAL = INTCONbits.GIE;
	INTCONbits.GIE = 0;
	EECON2 = 0x55;
	EECON2 = 0xAA;
	EECON1bits.WR = 1;
	while(EECON1bits.WR);				//Wait till the write completion
	INTCONbits.GIE = GIE_BIT_VAL;
	EECON1bits.WREN = 0;
}
//##############################################################################
//end VSK
