/**
 *******************************************************************
 *  Lesson 7 - "Reversible"
 *
 *  This lesson combines all of the previous lessons in using the button to reverse the
 *  direction of rotation when the button is pressed. The speed of rotation is controlled
 *  using the potentiometer.
 *
 *  The program needs to keep track of rotation direction and new code needs to be added
 *  to rotate in the other direction. Lesson 5 rotates right and checks for a '1' in the carry
 *  bit to determine when to restart the sequence. In Lesson 7, the program needs to rotate
 *  both ways and check for a '1' in bit 4 of the display when rotating to the left. When the
 *  '1' shows up in bit 4 of LATC, it will be re-inserted into bit  0 .
 *
 *  The debounce routine is more in-depth in this lesson because we need to keep in mind
 *  of the scenario of the switch being held down for long periods of time. If SW1 is held
 *  down, the LEDs would change direction rapidly, making the display look like it is out of
 *  control. The above flowchart will only change direction on the first indication of a solid
 *  press and then ignore the switch until it is released and pushed again. The switch must
 *  be pressed for at least the time it takes for the program to check the switch in its loop.
 *  Since the PIC MCU is running at 500 kHz, this will seem instantaneous.
 *
 *  LEDs will rotate at a speed that is proportional to the ADC value. The switch will toggle
 *  the direction of the LEDs
 *
 *  PIC: 18F14K22
 *  Compiler: XC8 v1.00  VSK 2.46
 *  IDE: MPLABX v1.10    VSK 6.20
 *
 *  Board: PICkit 3 Low Pin Count Demo Board
 *  Date: 6.1.2012       VSK 19.09.25
 *
 * *******************************************************************
 * See Low Pin Count Demo Board User's Guide for Lesson Information*
 * ******************************************************************
 */

//VSK #include <htc.h>                         //PIC hardware mapping
#include <xc.h>                          //PIC hardware mapping
#define _XTAL_FREQ 500000                //Used by the compiler for the delay_ms(x) macro

#define SWITCH_DOWN         0
#define SWITCH_UP           1

#define LED_RIGHT           1
#define LED_LEFT            0

#define SWITCH      PORTAbits.RA2

//VSK moved config bits to PICkit_3_Starter_Config.c

//prototype
unsigned char check_switch(void);
unsigned char adc(void);

//global variables
unsigned char _previous_state = SWITCH_UP; //if button is held down, we don't want to keep on switching directions


    /* -------------------LATC-----------------
     * Bit#:  -7---6---5---4---3---2---1---0---
     * LED:   ---------------|DS4|DS3|DS2|DS1|-
     *-----------------------------------------
     */

void main(void) {
    unsigned char delay;
    unsigned char direction;

    OSCCON = 0b00100010;                 //500KHz clock speed
    TRISC = 0;                           //all LED pins are outputs

    direction = LED_RIGHT;                //start from the right to left
    LATC = 0b00001000;                    //start with DS4 lit
    
    TRISAbits.TRISA4 = 1;                //Potentiamtor input
    TRISAbits.TRISA2 = 1;                //switch input
    ANSELbits.ANS2 = 0;                  //digital for switch
    ANSELbits.ANS3 = 1;                  //analog for pot
//VSK    ADCON0 = 0b00001101;                 //select RA4 as source of ADC, which is AN3, and enable the module
//VSK    ADCON2 = 0b00000001;                 //left justified - FOSC/8 speed
    ADCON1bits.PVCFG = 0;       //VDD
    ADCON1bits.NVCFG = 0;       //VSS (GND)
    ADCON2bits.ADFM = 0;        //left justified
    ADCON2bits.ACQT = 0b000;    //0TAD automatic aquisition time
    ADCON2bits.ADCS = 0b001;    //FOSC/8 speed  -> TAD 8/500kHz = 16us
    ADCON0bits.CHS = 3;
    ADCON0bits.ADON = 1;
    
    while (1) {

        delay = adc();                      //get the ADC value
        __delay_ms(5);                      //delay for AT LEAST 5ms
        while (delay-- != 0) __delay_ms(2); //decrement the 8 MSbs of the ADC and delay 2ms for each
        if (check_switch())                 //evaluates to be TRUE if switch was pressed
            direction ^= 1;                 //if pressed(not HELD!!), toggle the direction
        if (direction == LED_RIGHT) {
//VSK        LATC >> = 1;                     //shift to the right by 1
        LATC = LATC >> 1;                   //shift to the right by 1
           if(STATUSbits.CARRY)             //check for carry bit when rotating out of DS1
                LATCbits.LATC3 = 1;         //when the last LED is lit, restart the pattern
        } else{
//VSK            LATC << = 1;
            LATC = LATC << 1;
            if (LATCbits.LATC4 == 1)        //since rotating to the left with only 4 LEDs that are right justified, the LATC.LATC4
                LATCbits.LATC0 = 1;         //is bit tested instead. If this is set, restart the pattern

        }

    }
}

unsigned char check_switch(void) {
    if (SWITCH == SWITCH_DOWN) {            //switch is normally open to 5V ...when pressed, RA4 is connected to GND
        __delay_ms(5);                      //debounce by delaying and checking if switch is still pressed
        if(_previous_state == SWITCH_DOWN && SWITCH == SWITCH_DOWN)
            return 0;                       //don't toggle
        else if (SWITCH == SWITCH_DOWN) {   //check if still down
            _previous_state = SWITCH_DOWN;
            return 1;                       //reverse the LEDs
        } else{
            _previous_state =SWITCH_UP;     //switch not properly pressed
            return 0;                       //don't reverse
        }
    } else{
        _previous_state = SWITCH_UP;        //switch not pressed
        return 0;                           //don't reverse
    }
}

unsigned char adc(void) {
//VSK    __delay_us(5);                  //wait for ADC charging cap to settle
//VSK always same channel here and function call waste lot of time anyway
    ADCON0bits.GO = 1;
    while (ADCON0bits.NOT_DONE) continue;//wait for conversion to be finished
    return ADRESH;
}