/**
 *******************************************************************
 * Lesson 11 - "Indirect addressing"
 *
 * This lesson covers a very important topic of indirect addressing. The code uses indirect
 * addressing to implement a moving average filter. This lesson adds a moving average
 * filter to the Analog-to-Digital code in Lesson 4. The moving average keeps a list of the
 * last ADC values (n) and averages them together. The filter needs two parts: A circular
 * queue and a function to calculate the average.
 *
 * Twisting the potentiometer changes the value read by the Analog-to-Digital converter.
 * The filtered value is then sent to the LED display.
 *
 * This lesson provides the same outcome as Lesson 4. The user rotates the POT to see
 * the LEDs rotate. The top four MSbs of the ADC value are reflected onto the LEDs.
 *
 * PIC: 18F14K22
 * Compiler: XC8 v1.00  VSK 2.46
 * IDE: MPLABX v1.10    VSK 6.20
 *
 * Board: PICkit 3 Low Pin Count Demo Board
 * Date: 6.1.2012       VSK 19.09.25
 *
 * *******************************************************************
 * See Low Pin Count Demo Board User's Guide for Lesson Information*
 * ******************************************************************
 */

//VSK #include <htc.h>                         //PIC hardware mapping
#include <xc.h>                         //PIC hardware mapping
#define _XTAL_FREQ 500000               //Used by the compiler for the delay_ms(x) macro

#define NUM_READINGS 8                  //number to divide the total

//VSK moved config bits to PICkit_3_Starter_Config.c

//prototypes
unsigned char average(unsigned char *ptr);
unsigned char adc(void);

//globals
unsigned int _sum = 0;                  //keeps the average value of the ADC result

    /* -------------------LATC-----------------
     * Bit#:  -7---6---5---4---3---2---1---0---
     * LED:   ---------------|DS4|DS3|DS2|DS1|-
     *-----------------------------------------
     */

void main(void) {
    unsigned char *ptr_queue;
    unsigned char queue[8] = {0};       //keeps the last 8 ADC readings
    unsigned char i;

    OSCCON = 0b00100010;                //500KHz clock speed
    TRISC = 0;                          //all LED pins are outputs

    TRISAbits.TRISA4 = 1;                //Potentiamtor is connected to RA4...set as input
    ANSELbits.ANS3 = 1;                  //analog input - different than pic16 syntax
    ADCON0 = 0b00001101;                 //select RA4 as source of ADC, which is AN3, and enable the module
    ADCON2 = 0b00000001;                 //left justified - FOSC/8 speed

    while (1) {
        ptr_queue = &queue;             //point to the first byte in this array (RESET the pointer)
        for (i = NUM_READINGS; i != 0; i--){
            LATC = (average(ptr_queue) >> 4 ); //only want the 4 MSbs for 4 LEDs
            ptr_queue++;
        }
    }
}

unsigned char average(unsigned char *ptr) {
    unsigned char adc_value;

    _sum -= *ptr;                       //subtract the current value out of the sum
    adc_value = adc();
    *ptr = adc_value;                   //assign ADC value to the queue
    _sum += adc_value;                  //add to the sum
    
    return (_sum/NUM_READINGS);         //compute the average
}

unsigned char adc(void) {

    return ADRESH;                      //grab the top 8 MSbs
}
