   TITLE    "18F14K22_LCD_2wire_64MHz_Test.INC"
   NOLIST   ;don't show this text in the <actual.asm>.LST-File
;--------------------------------------------------------------------
; Filename: 18F14K22_LCD_2wire_64MHz_Test.INC
; Author:   Ottmar
; Descrip:  Using HD44780 LCD in 2bit-mode with 74LS164 adaptor
; Created:  2014.01.08
; Updated:  2023.05.13
; Version:  09
;*********************************************************************
;-SUBROUTINES -   not intended for direct call!
; Control8Bit:    with 8-bit interface (internal only)
; Clear_ShiftReg: Clears the shift register
; Delay:          1.5ms delay between LCD commands
; OutLcd:         Output of the upper/lower nibble (internal only)
;
;-SUBROUTINES - can be called from other code
; Lcd_Init:       Initialization of HD44780 LCD with 4-bit interface
; OutDDRAM_Addr:  Output a DDRAM address to the LCD
; OutLcd_Ctrl:    Output of an instruction to the LCD
; OutLcd_Ascii:   Output of an Ascii character, value in WREG
;                 e.g. movlw .35 -> output '[' ROM Code A00!
; OutLcd_Data:    Output of a single character value in the WREG
;                 e.g. movfw BCD9 (may only contain 1 character!"
; Lcd_Clear:      Clears the text in the LCD
; Lcd_Ready:      Functional test! Output of "LCD 2Wire READY!"
; CGRAM_Load:     Load self-defined character into the LCD
;
; Delay1ms:       delays   1.000wc
; Delay10ms;      "       10.000wc
; Delay100ms:     "      100,000wc   
; Delay1000ms     "    1.000.000wc
;*********************************************************************
; The initialization and operation of an LCD with Fosc 4 - 64MHz
; is successfully tested. In the section "LABELS and CONSTANTS"
; the current value of Fosc must be entered!
;*********************************************************************
; ICSP - ATTENTION!
; If the ICSP interface is used, the pins RB6 (PGC) AND RB7 (PGD) must
; be decoupled from the rest of the circuit, with a resistance of at
; least 1 kOhm (Affects also LCD data lines ).
;
; If this does not happen, the PICkit3 signals when connecting:
;    "Target Device ID (00000000) does not match expected Device
;     ID (00001060)"
;     oder
;    "The following memory regions failed to program correctly:
;     Program Memory"
;*********************************************************************
; HD44780 kompatible Controller:
;   KS0066     KS0073     KS007B     KS0076B
;   SED1278F   S6A0069    ST7066     SPLC780A1
;*********************************************************************
; LCD (HD44780)   RW=0     Write only! Pin isconnected to GND
;                 RS=0/1   Register select Instruction/Data
;                 E        toggle H->L takes overt DB7:DB4 in Data- or
;                          Instructionregister
;*********************************************************************
   LIST
   #IFNDEF  FOSC_MULT
      FOSC        EQU    d'16'      ;actual fosc(MHz)   
      FOSC_MULT   EQU    FOSC/d'4'  ;4 clocks of fosc = 1 wc   
   #ENDIF
   ;
;--LCD-Initialization starts here
Lcd_Init:
   BANKSEL  0
   movlw    d'5'*FOSC_MULT ;50ms delay
   CALL     Delay10ms      ;gives time for Vdd to rise up to >4.5V
   ;
   movlw    b'00110000'    ;1. Function set
   CALL     Control8Bit
   
   movlw    d'5'*FOSC_MULT ;  50ms
   CALL     Delay10ms
   ;
   movlw    b'00110000'    ;2. Function set
   CALL     Control8Bit    ;   (Interface is 8 bits long.)   
   ;
   movlw    b'00110000'    ;2. Function set
   CALL     Control8Bit    ;   (Interface is 8 bits long.)   
   ;
   movlw    b'00110000'    ;3. Function set
   CALL     Control8Bit    ;   (Interface is 8 bits long.)
   ;
   movlw    b'00100000'    ;4. Function set
   CALL     Control8Bit    ;   INTERFACE IS NOW 4 BITS LONG. 
   ;
   movlw    b'00101000'    ;5. Function set 2 AND MORE LINES 5x8 Dot
   CALL     OutLcd_Ctrl ;      send function to LCD
   ;
   movlw    b'00001000'    ;6. display off
   CALL     OutLcd_Ctrl
   ;
   movlw    b'00000110'    ;7. entry mode, increment
   CALL     OutLcd_Ctrl ;      disable display-shift
   ;
   movlw    b'00000011'    ;8. cursor home, cursor home
   CALL     OutLcd_Ctrl
   ;
   movlw    b'00001100'    ;9. Set   Display ON, Cursor OFF Blink OFF
   CALL     OutLcd_Ctrl ;            b3      b2         b1          b0
   ;
   movlw    b'00000001'    ;   clear Display lschen + cusor home
   CALL     OutLcd_Ctrl
   RETURN
;*********************************************************************
Control8Bit:
; Called only at the beginning of the LCD initialization, as long as
; the 8-bit interface has not yet been converted to 4-bit.
; LCD_RS=0 causes RS to be set to 0 and thus accepted into the LCD
; INSTRUCTIONS tab.
; NIBBLE=1 causes the upper/lower nibble to be output.
; 
;  Handing over in WREG:   b'00110000'
   bcf      LCD_RS         ;LcdFlags,7=0? RS=0 Instruction
   bcf      NIBBLE         ;LcdFlags,6=0  send only upper nibble
   CALL     OutLcd         ;
   RETURN
;*********************************************************************
Clear_ShiftReg:
; Before each output of a nibble into the shift register, its
; Outputs Q5:Q0 are set to low level to prevent toggling of
; LCD enable using the DATA line. This is done by setting the DAT
; line to low and the L level pushed through 6 times, so that
; Q5:0 = '000000" is present.
   ;
   movlw    .6             ;Preset bit counter (6 outputs Q5:Q0)
   movwf    LcdBit         ;
   bcf      LCD_DATA       ;b0 = 0
clearshiftr1:
   bsf      LCD_CLK        ;LCD_PORT,b1 H->L RS -> ShiftReg
   nop
   bcf      LCD_CLK        ;transfers LCD_DATA to the shift register
   BANKSEL  0
   decf     LcdBit,f       ;mov 6 xLOW in ShiftReg 
   btfss    STATUS,Z       ;Z=1 done?
   GOTO     clearshiftr1   ;No do it again
   RETURN
;*********************************************************************
;  SUBROUTINES FOR CALL FROM OUTSIDE
;  END OF TRANSLATION TO ENGLISH
;*********************************************************************
OutCGRAM_Addr:
; WREG: Transfer of the CGRAM address in which the user-defined
;       character is to be loaded
   andlw    b'00111111'    ;masking b7:6
   iorlw    SET_CGRAM_ADDR ;Link with function "SET CGRAM"
   GOTO     OutLcd_Ctrl
   ;
OutDDRAM_Addr:
; WREG: bergabe der LCD-DDRAM-Adresse in welcher das Zeichen
;ausgegeben werden soll (z.B. movlw LINEx+Digit movlw LINE2+.5)
   iorlw    SET_DDRAM_ADDR  ;Adresse mit Funktionsbefehl verknpfen
   ;
OutLcd_Ctrl:
; LCD_RS =0   bewirkt die bernahme in das LCD-INSTRUCTIONS-Register
; NIBBLE =1/0 bewirkt die Ausgabe des oberen/unteren Nibble.
   ;
   bcf      LCD_RS         ;LcdFlags,7=0 RS=0 Instruction
   bsf      NIBBLE         ;LcdFlags,6=1/0 Oberes, Unteres Nibble
   CALL     OutLcd         ;Oberes+unteres Nibble ausgeben
   RETURN
;*********************************************************************
OutLcd_Ascii:              ;addiert zum WREG .48 -> Asciicode     
   addlw    .48            ;0-9..a-z..A-Z
   ;
OutLcd_Data:              
; Aufruf:  movlw "X"          X-> beliebiges ASCII-Zeichen
;          CALL   OutLcdData
; LCD_RS=1 Datenbefehl folgt - bernahme in das LCD-DATEN-Register.
; NIBBLE=1 Ausgabe des oberen/unteren Nibble in das Schieberegister
   bsf      LCD_RS         ;LcdFlags,7=1 RS=1 Datenausgabe
   bsf      NIBBLE         ;LcdFlags,6=1 Oberes, Unteres Nibble
   CALL     OutLcd         ;Oberes+unteres Nibble ausgeben
   RETURN
;*********************************************************************
OutLcd:
; Ist das 8Bit-Interface noch aktiv, wird nur das obere Nibble in das
; Instruktionsregister bernommen (NIBBLE=0, LCD_RS=0).
; Ist das 4Bit-Interface aktiviert, werden das Obere UND das Untere
; Nibble in das Shiftregister bertragen (NIBBLE=1). 
   ;
;--OBERES bzw. UNTERES NIBBLE bertragen
   movwf    LcdData        ;zu bertragendes Byte  vom WREG in die
   ;                       ;Arbeitsvariable kopieren
outlcd_1:
   CALL     Clear_ShiftReg ;Setzt Q5:Q0 auf Low-Pegel (42cycles)
;--1. ENABLE-Bit
   bsf      LCD_DATA       ;LCD_PORT,0 Enable=H (->Q5)
   bsf      LCD_CLK        ;LCD_PORT,1 H->L bernimmt DATA ins ShiftReg
   ;
   bcf      LCD_CLK
;--2. RS-Bit DATENREGISTER
   btfsc    LCD_RS         ;LcdFlags,7=1? RS=1? Datenausgabe?
   bsf      LCD_DATA       ;JA, RS=1 (->Q4)
   btfss    LCD_RS         ;LcdFlags,7=0? RS=0? Instruktion ausgeben?
   bcf      LCD_DATA       ;JA, RS=0 (->Q4)   
   bsf      LCD_CLK
   bcf      LCD_CLK        ;CLOCK H->L bernahme DATA ins ShiftReg
;--3. 4 Bit (Nibble) LCD-Code 
   movlw    .4             ;Wert fr 4 Bits/Nibble
   movwf    LcdBit         ;in Zhler kopieren
outlcd_2:
   btfss    LcdData,7      ;Bit 7 = Low?-wird ja nicht verwendet
   bcf      LCD_DATA       ;JA, LCD_PORT,0 auf Low-Pegel setzen
   btfsc    LcdData,7      ;Bit 7 =High?
   bsf      LCD_DATA       ;JA, LCD_PORT,0 auf High-Pegel setzen
   bsf      LCD_CLK        ;mit LCD_PORT,CLK H->L Pegel an DATA in das
   bcf      LCD_CLK        ;    Schieberegister bernehmen
   clrf     STATUS,C
   RLNCF    LcdData,f
;   rlf      LcdData,f      ;LCD bit6->B7 Rotate Left f (No Carry)
   decf     LcdBit,f       ;Bitzhler - 1 = 0?
   btfss    STATUS,Z
   GOTO     outlcd_2       ;NEIN, nchstes Bit ins Schieberegister
   ;
;--4. Nibble ins LCD bernehmen 
   bsf      LCD_DATA       ;DATA-Leitung H toggeln dabei wird
   nop                     ;das Nibble mit der HL-Fanke ins LCD
   nop                     ;bernommen
   bcf      LCD_DATA
   ;
   #IFNDEF  __DEBUG
      movlw    .1*FOSC_MULT   ;Anpassung an aktuelle fosc
      CALL     Delay          ;Verarbeitungszeit im LCD vor bertragung
                              ;des nchsten Nibble abwarten (1541 cycles)
      btfss    NIBBLE         ;LcdFlags,6=0? Beide Nibble ausgegeben?
      GOTO     outlcd_0       ;JA, Fertig!
      bcf      NIBBLE         ;NEIN, oberes Nibble ist ausgegeben
      GOTO     outlcd_1       ;jetzt noch unteres Nibble ausgeben
   #ENDIF
outlcd_0:
   RETURN
;*********************************************************************
Lcd_Clear:
   movlw    b'00000001'    ;Display Clear
   CALL     OutLcd_Ctrl
   RETURN
;*********************************************************************
Lcd_Ready:
;after calling "LcdReady" you need to add a delay resp.:
;           movlw   d'20'*FOSC_MULT
;           CALL   Delay100ms   ;displaya text during 20x100ms
   ;
   movlw    LINE1          ;Ausgabe in Zeile 1, Digit 0
   CALL     OutDDRAM_Addr  ;send instruction to LCD
   movlw    "L"            ;Digit 0, copy character as ansii-code
   CALL     OutLcd_Data    ;to WREG send charactercode to LCD
   movlw    "C"
   CALL     OutLcd_Data
   movlw    "D"
   CALL     OutLcd_Data
   movlw    " "
   CALL     OutLcd_Data
   movlw    "2"
   CALL     OutLcd_Data
   movlw    "W"
   CALL     OutLcd_Data
   movlw    "i"
   CALL     OutLcd_Data
   movlw    "r"            ;Digit 7
   CALL     OutLcd_Data
   ;
;--Bedingte Kompilierung:
;   Bei LCD 2x8 (LCD_TYPE=1) wird der Testtext 2zeilig zu je 8 Zeichen
;   ausgegeben. Bei allen anderen LCDs (LCD_TYPE > 1) erfolgt die Ausgabe 
;   1zeilig
   #IFNDEF  LCD_TYPE       ;falls nicht definert, Ausgabeformat
      LCD_TYPE    EQU .1   ;fr LCD 2x8 verwenden
      LCD_LINES   EQU .2
      LCD_DIGITS  EQU .8
   #ENDIF
   ; 
   IF LCD_DIGITS == 8   ;LCD 2zeilig, zu je 8 Digits (2x8)?
      movlw LINE2          ;JA, Ausgabe in DDRAM-Adresse Zeile 2
      CALL  OutDDRAM_Addr  ;Send instruction to LCD
   ENDIF
   ;
   movlw    "e"            ;Nur bei LCD 2x8: Ausgabe in Zeile 2
   CALL     OutLcd_Data    ;sonst: weiter mit Ausgabe in Digit 8
   movlw    " "
   CALL     OutLcd_Data
   movlw    "R"
   CALL     OutLcd_Data
   movlw    "e"
   CALL     OutLcd_Data
   movlw    "a"
   CALL     OutLcd_Data
   movlw    "d"
   CALL     OutLcd_Data
   movlw    "y"
   CALL     OutLcd_Data
   movlw    "!"
   CALL     OutLcd_Data     ;Digit 15 (2x8 Line 2, Digit 7)
   RETURN
;*********************************************************************
Delay:      ;1520 cycles Wartezeit zur Verarbeitung von 1 Nibble
; Aufruf: 
; movlw   Literal*FOSC_MULT ;Literal = (Zahl*FOSC_MULT) <=255
; CALL    Delay100ms        ;FOSC_MULT vgl. ';LABELS and CONSTANTS'; 1,52 ms Delay nach jeder bertragung eines Nibble. 
   ;
   movwf    d3
   xorlw    .0
   btfsc    STATUS,Z    ;Z=1? WREG=0? 
   incf     d3,f        ;JA, d3+1 -> 10ms Delayzeit   
   ;   
delay_2:
      movlw .1
      movwf d2         
delay_1:   
   movlw    .150
   movwf    d1
delay_0:
   nop      ;6 x nop=10 cycles/durchlauf
   nop
   nop
   nop
   nop   
   nop      
   decf     d1,f
   btfss    STATUS,Z
   GOTO     delay_0
  
   decf     d2,f
   btfss    STATUS,Z
   GOTO     delay_1
   
   decf     d3,f
   btfss    STATUS,Z
   GOTO     delay_2          
   RETURN
;*********************************************************************
Delay1ms:
; 1000 working cycles incl. CALL
; Aufruf: movlw   Literal*FOSC_MULT ;Literal = (Zahl*FOSC_MULT) <=255
;         CALL    Delay1ms        ;FOSC_MULT vgl. ';LABELS and CONSTANTS'
   ;
   movwf    d3          ;Multiplikator fr Basisdelay
   xorlw    .0          ;Vergleich WREG = d3 = 0 
   btfsc    STATUS,Z    ;Z=1? WREG=0? 
   incf     d3,f        ;JA, d3+1 -> 10ms Delayzeit
   ;
delay1ms2:
   movlw    .13
   movwf    d2
delay1ms1:          
   movlw    .10
   movwf    d1
delay1ms0:
   nop      ;6 x nop=10 cycles/durchlauf
   nop
   nop
   decf     d1,f
   btfss    STATUS,Z
   GOTO     delay1ms0
   nop
   decf     d2, f
   btfss    STATUS,Z
   GOTO     delay1ms1
   decf     d3,f
   btfss    STATUS,Z
   GOTO     delay1ms2
   RETURN
;*********************************************************************
Delay10ms:
; 10032 working cycles incl. CALL
; Aufruf: movlw   Literal*FOSC_MULT ;Literal = (Zahl*FOSC_MULT) <=255
;         CALL    Delay10ms        ;FOSC_MULT vgl. ';LABELS and CONSTANTS'
   movwf    d3
   xorlw    0x00
   btfsc    STATUS,Z
   incf     d3,f
delay10ms2
   movlw    .20
   movwf    d2   
delay10ms1:
   movlw    .124;.125
   movwf    d1
delay10ms0:
   decf     d1,f
   btfss    STATUS,Z
   GOTO     delay10ms0
   decf     d2,f
   btfss    STATUS,Z 
   GOTO     delay10ms1
   decf     d3,f
   btfss    STATUS,Z 
   GOTO     delay10ms2
   RETURN
;*********************************************************************
Delay100ms:
; 100.002 working cycles inclusive call
; Aufruf: movlw   Zahl*FOSC_MULT ;Wert = Zahl von 1-255
;         CALL    Delay100ms     ;FOSC_MULT vgl. ';LABELS and CONSTANTS'
   ;   ;
   movwf    d4
   xorlw    0x00
   btfsc    STATUS,Z
   incf     d4,f

delay100ms3:
   movlw    .11
   movwf    d3
delay100ms2:
   movlw    .10
   movwf    d2
delay100ms1:
   movlw    .100;
   movwf    d1
delay100ms0:
   nop
   nop
   nop
   nop
   nop
   decf     d1,f
   btfss    STATUS,Z
   GOTO     delay100ms0
   nop
   nop
   nop
   decf     d2,f
   btfss    STATUS,Z 
   GOTO     delay100ms1
   nop
   nop
   nop
   nop 
   nop  
   decf     d3,f
   btfss    STATUS,Z 
   GOTO     delay100ms2
   decf     d4,f
   btfss    STATUS,Z 
   GOTO     delay100ms3
   ;
   RETURN 
;*********************************************************************
Delay1000ms:
; 1.000.207 working cycles inclusive call
; Aufruf: movlw   Zahl*FOSC_MULT ;Wert = Zahl von 1-255
;         CALL    Delay100ms     ;FOSC_MULT vgl. ';LABELS and CONSTANTS'
   ;   ;
   movwf    d4          ;Delay = d3 * 100ms
   xorlw    .0          ;d3=0 ist verboten, dann d3=1 setzen
   btfsc    STATUS,Z    ;Z=1? d3=0? 
   incf     d4,f        ;JA, d3+1 -> 100ms Delayzeit
delay1s_3
   movlw    .7
   movwf    d3
delay1s_2:
   movlw    .80;
   movwf    d2
delay1s_1:
   movlw    .178 
   movwf    d1
delay1s_0:
   nop
   nop
   nop
   nop
   nop
   nop
   decf     d1, f
   btfss    STATUS,Z
   GOTO     delay1s_0

   decf     d2, f
   btfss    STATUS,Z
   GOTO     delay1s_1   
   nop
   decf     d3,f
   btfss    STATUS,Z
   GOTO     delay1s_2

   decf     d4,f
   btfss    STATUS,Z
   GOTO     delay1s_3
   ;
   RETURN
;*********************************************************************

